'use strict';

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

/* --------------------------------------------------------------
 product_cart_handler.js 2025-01-17
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2025 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Component that includes the functionality for
 * the add-to-cart, refresh and delete buttons
 * on the wishlist and cart
 */
gambio.widgets.module('product_cart_handler', ['form', 'xhr', gambio.source + '/libs/events', gambio.source + '/libs/modal.ext-magnific', gambio.source + '/libs/modal'], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        $window = $(window),
        $body = $('body'),
        $form = null,
        $updateTarget = null,
        $deleteField = null,
        $cartEmpty = null,
        $cartNotEmpty = null,
        deleteFieldName = null,
        action = null,
        busy = null,
        updateList = false,
        transition = null,
        active = {},
        isChanged = false,
        defaults = {
        // Use an AJAX to update the form
        ajax: true,
        // Show an confirm-layer on deletion of an item
        confirmDelete: false,
        // Selector of the hidden field for the deletion entries
        deleteInput: '#field_cart_delete_products_id',
        // Trigger an event to that item on an successfull ajax (e.g. the shipping costs element)
        updateTarget: '.shipping-calculation',
        // The URL for the quantity check of the item
        checkUrl: 'shop.php?do=CheckQuantity',
        // If an URL is set, this one will be requests for status updates on tab focus
        updateUrl: 'shop.php?do=Cart',

        changeClass: 'has-changed', // Class that gets added if an input has changed
        errorClass: 'error', // Class that gets added to the row if an error has occured
        cartEmpty: '.cart-empty', // Show this selection if the cart is empty or hide it else
        cartNotEmpty: '.cart-not-empty', // Show this selection if the cart is not empty or hide it else
        classLoading: 'loading', // The class that gets added to an currently updating row
        actions: { // The actions that getting appended to the submit url on the different type of updates
            add: 'wishlist_to_cart',
            delete: 'update_product',
            refresh: 'update_wishlist'
        },
        ajaxActions: { // URLs for the ajax updates on the different actions
            add: 'shop.php?do=WishList/AddToCart',
            delete: 'shop.php?do=Cart/Delete',
            refresh: 'shop.php?do=Cart/Update'
        },
        selectorMapping: {
            buttons: '.shopping-cart-button',
            giftContent: '.gift-cart-content-wrapper',
            giftLayer: '.gift-cart-layer',
            shareContent: '.share-cart-content-wrapper',
            shareLayer: '.share-cart-layer',
            hiddenOptions: '#cart_quantity .hidden-options',
            message: '.global-error-messages',
            infoMessage: '.info-message',
            shippingInformation: '#shipping-information-layer',
            totals: '#cart_quantity .total-box',
            errorMsg: '.error-msg',
            submit: '.button-submit'
        }
    },
        options = $.extend(false, {}, defaults, data),
        module = {};

    // ########## HELPER FUNCTIONS ##########

    /**
     * Updates the form action to the type given
     * in the options.actions object
     * @param       {string}        type        The action name
     * @private
     */
    var _setAction = function _setAction(type) {
        if (options.ajax) {
            action = options.ajaxActions[type];
        } else if (options.actions && options.actions[type]) {
            action = action.replace(/(action=)[^\&]+/, '$1' + options.actions[type]);
            $form.attr('action', action);
        }
    };

    /**
     * Helper function that updates the
     * hidden data attributes with the current
     * values of the input fields
     * @param       {object}        $target     jQuery selection of the topmost container
     * @private
     */
    var _updateDataValues = function _updateDataValues($target) {
        $target.find('input[type="text"]').each(function () {
            var $self = $(this),
                value = $self.val();

            $self.data('oldValue', value);
        });
    };

    /**
     * Helper function that restores the values
     * stored by the _updateDataValues function
     * @param       {object}        dataset     The data object of all targets that needs to be reset
     * @private
     */
    var _restoreDataValues = function _restoreDataValues(dataset) {
        // Reset each changed field given
        // by the dataset target
        $.each(dataset, function () {
            var value = this;

            value.target.find('.' + options.changeClass).each(function () {
                var $self = $(this),
                    name = $self.attr('name').replace('[]', ''),
                    val = $self.data().oldValue;

                value[name][0] = val;
                $self.val(val).removeClass(options.changeClass);
            });
        });
    };

    /**
     * Helper function that generates an array of  datasets from the form. Each array item
     * contains the data of one row (inclusive the attributes data from the form head belonging
     * to the row). Additionally it adds the target-parameter to each dataset which contains
     * the selection of the row,the current dataset belongs to.
     *
     * @param {object} $row The optional row selection the data gets from. If no selection is given, the form
     * gets selected.
     * @return {Array} The array with the datasets of each row
     *
     * @private
     */
    var _generateFormdataObject = function _generateFormdataObject($row) {
        var $target = $row && $row.length ? $row : $form,
            $rows = $row && $row.length ? $row : $form.find('.order-wishlist .item:gt(0)'),
            $hiddens = $form.find('.hidden-options input[type="hidden"]'),
            dataset = jse.libs.form.getData($target),
            result = [],
            tmpResult = null;

        $.each(dataset.products_id, function (i, v) {
            tmpResult = {};
            tmpResult.target = $rows.eq(i);

            // Store the data from the current row as a json
            $.each(dataset, function (key, value) {
                if ((typeof value === 'undefined' ? 'undefined' : _typeof(value)) === 'object' && value[i] !== undefined) {
                    // Store the value as an array to be compliant with the old API
                    tmpResult[key] = [value[i]];
                }
            });

            // Get the hidden fields for the attributes
            // belonging to this row from the form head
            $hiddens.filter('[name^="id[' + v + '"], .force').each(function () {
                var $self = $(this),
                    name = $self.attr('name');

                tmpResult[name] = $self.val();
            });

            // Push the generated json to the final result array
            result.push(tmpResult);
        });

        return result;
    };

    /**
     * Function that checks the form / the row if the combination
     * and quantity is valid. It returns an promise which gets rejected
     * if in the scope was an invalid value. In other cases it gets
     * resolved. If it is detecting changes inside the form it can
     * show an info layer to the user and / or revert the changes
     * (depending on the caller parameters)
     * @param       {boolean}       showChanges         Show an info-layer if changes would be refused
     * @param       {boolean}       revertChanges       Resets the form values with the one from the data attributes if true
     * @param       {object}        formdata            Json that contains the data to check
     * @return     {*}                                 Returns a promise
     * @private
     */
    var _checkForm = function _checkForm(showChanges, revertChanges, formdata) {

        var promises = [],
            hasChanged = false;

        // Get the complete form data if no row data is given
        formdata = formdata || _generateFormdataObject();

        // Check the formdata for changed values
        $.each(formdata, function () {
            var $changed = this.target.find('.' + options.changeClass);
            hasChanged = hasChanged || !!$changed.length;
            return !hasChanged;
        });

        return $.when.apply(undefined, promises).promise();
    };

    /**
     * Helper function that cleans up the process state
     * (Needed especially after ajax requests, to be able
     * to make further requests)
     * @param       {string}        id              The product id that needs to be reseted
     * @return     {Array.<T>}                     Returns an array without empty fields
     * @private
     */
    var _cleanupArray = function _cleanupArray(id, $row) {
        delete active['product_' + id];
        $row.removeClass('loading');
        return active;
    };

    /**
     * Helper function that does the general form update
     * after an ajax request
     * @param       {object}    $target         The jQuery selection of the target elements.
     * @param       {object}    result          The result of the ajax request.
     * @param       {string}    type            The executed action type.
     * @private
     */
    var _updateForm = function _updateForm($target, result, type) {
        // Update the rest of the page
        jse.libs.theme.helpers.fill(result.content, $body, options.selectorMapping);

        // Toggle info-messages visibility.
        $('.info-message').toggleClass('hidden', $('.info-message').text() === '');

        // Inform other widgets about the update
        $updateTarget.trigger(jse.libs.theme.events.CART_UPDATED(), []);
        $body.trigger(jse.libs.theme.events.CART_UPDATE(), type === 'add');

        // Update the hidden data attributes of that row
        _updateDataValues($target);

        if ($.isEmptyObject(result.products)) {
            // Hide the table if no products are at the list
            $cartNotEmpty.addClass('hidden');
            $cartEmpty.removeClass('hidden');
        } else {
            // Show the table if there are products at it
            $cartEmpty.addClass('hidden');
            $cartNotEmpty.removeClass('hidden');
        }

        // reinitialize widgets in updated DOM
        window.gambio.widgets.init($this);
    };

    /**
     * Helper function to update product specific messages.
     * Therefor it calls AJAX-requests (in case ajax is
     * enabled) to the server to get the updated information
     * about the table state. If ajax isn't enabled, it simply
     * submits the form.
     * @param       {object}        messages            Object with articleId => message configuration object
     * @private
     */
    var _updateArticlesMessage = function _updateArticlesMessage(messages) {

        //update all messages og the request
        $.each(messages, function (product, message) {
            var messageCfg = { 'message': message };
            //let productId = product.match(/\d+/)[0];
            // Find the related product tr
            var item = $('input[value^="' + product + '"]').closest('tr');

            jse.libs.theme.helpers.fill(messageCfg, item, options.selectorMapping);
            if (item.find('.error-msg').text() !== '') {
                item.find('.error-msg').show();
            }
        });
    };

    /**
     * Helper function that processes the list updates.
     * Therefor it calls AJAX-requests (in case ajax is
     * enabled) to the server to get the updated information
     * about the table state. If ajax isn't enabled, it simply
     * submits the form.
     * @param       {object}        $target            The jQuery selection of the row that gets updated
     * @param       {object}        dataset            The data collected from the target row in JSON format
     * @param       {article}       article            The products id of the article in that row
     * @param       {article}       type               The operation type can either be "add", "delete" or "refresh".
     * @private
     */
    var _executeAction = function _executeAction($row, $target, dataset, article, type) {
        if (options.ajax) {
            // Delete the target element because ajax requests
            // will fail with a jQuery selection in the data json
            delete dataset.target;

            $row.trigger(jse.libs.theme.events.TRANSITION(), transition);

            // Perform an ajax if the data is valid and the options for ajax is set
            jse.libs.xhr.ajax({ url: action, data: dataset }, true).done(function (result) {
                // Perform hooks
                jse.libs.hooks.execute(jse.libs.hooks.keys.shop.cart.change, {
                    $target: $target,
                    dataset: dataset,
                    article: article,
                    type: type,
                    result: result
                }, 500);

                // Update the product row
                var $markup = $(result.products['product_' + article] || '');

                // Toggle error-messages visibility.
                $markup.removeClass(options.classLoading);
                $target.replaceWith($markup);

                _updateArticlesMessage(result.content.errorMessageList);
                delete result.content.errorMessageList;

                _updateForm($target, result, type);

                var productNumber = article.match(/\d+/)[0];

                // Find all items with the same product number
                var $items = $('input[value^="' + productNumber + '"]').parent('td');

                // Apply the new markup foreach item which has the same product number.
                $items.each(function () {
                    if (!$(this).find('input[value="' + article + '"]').length) {
                        var number = $(this).find('input[id="products_id[]"]').attr('value');
                        $markup = $(result.products['product_' + number] || '');
                        $target = $(this).parent('tr');
                        $target.replaceWith($markup);
                    }
                });
            }).always(function () {
                _cleanupArray(article, $row);
            });
        } else {
            // Cleanup the active array on fail / success
            // of the following submit. This is a fallback
            // if an other component would prevent the submit
            // in some cases, so that this script can perform
            // actions again
            var deferred = $.Deferred();
            deferred.always(function () {
                _cleanupArray(article, $row);
            });

            // Submit the form
            $form.trigger('submit', deferred);
        }
    };

    // ########## EVENT HANDLER ##########

    /**
     * Adds an class to the changed input
     * field, so that it's styling shows
     * that it wasn't refreshed till now
     * @private
     */
    var _inputHandler = function _inputHandler() {
        var $self = $(this),
            value = parseInt($self.val()),
            oldValue = parseInt($self.data().oldvalue),
            hasNewValue = value !== oldValue;

        if (hasNewValue) {
            isChanged = hasNewValue;
            $self.addClass(options.changeClass);
        } else {
            $self.removeClass(options.changeClass);
        }

        _updateChangeState();
    };

    /**
     * Handle the blur event
     * @private
     */
    var _blurHandler = function _blurHandler() {
        var $self = $(this),
            value = parseInt($self.val()),
            oldValue = parseInt($self.data().oldvalue),
            hasNewValue = value !== oldValue;

        if (hasNewValue) {
            $self.closest('.item').find('.button-refresh').first().trigger('click');
        }
    };

    /**
     * Handler that listens on click events on the
     * buttons "refresh", "delete" & "add to cart".
     * It validates the form / row and passes the
     * the data to an submit execute funciton if valid
     * @param       {object}    e       jQuery event object
     * @private
     */
    var _clickHandler = function _clickHandler(e) {
        e.preventDefault();
        e.stopPropagation();

        var $self = $(this),
            $row = $self.closest('.item'),
            type = e.data.type,
            rowdata = _generateFormdataObject($row)[0],
            article = rowdata.products_id[0],
            $target = rowdata.target,
            title = $target.find('.product-title').text();

        // Add loading class
        $row.addClass('loading');

        // Check if there is no current process for this article
        // or in case it's no ajax call there is NO other process
        if ($.isEmptyObject(active) || options.ajax && !active['product_' + article]) {
            active['product_' + article] = true;
            _setAction(type);

            switch (type) {
                case 'delete':
                    // Update the form and the dataset with
                    // the article id to delete
                    $deleteField.val(article);
                    rowdata[deleteFieldName] = [article];

                    if (options.confirmDelete) {
                        // Open a modal layer to confirm the deletion
                        var modalTitle = jse.core.lang.translate('CART_WISHLIST_DELETE_TITLE', 'general'),
                            modalMessage = jse.core.lang.translate('CART_WISHLIST_DELETE', 'general');

                        jse.libs.theme.modal.confirm({
                            content: modalMessage,
                            title: modalTitle
                        }).done(function () {
                            var deferred = $.Deferred();

                            deferred.done(function () {
                                _executeAction($row, $target, rowdata, article, type);
                            });

                            $body.trigger(jse.libs.theme.events.WISHLIST_CART_DELETE(), [{
                                'deferred': deferred,
                                'dataset': rowdata
                            }]);
                        }).fail(function () {
                            _cleanupArray(article, $row);
                        });
                    } else {
                        var deferred = $.Deferred();

                        deferred.done(function () {
                            _executeAction($row, $target, rowdata, article, type);
                        });

                        $body.trigger(jse.libs.theme.events.WISHLIST_CART_DELETE(), [{
                            'deferred': deferred,
                            'dataset': rowdata
                        }]);
                    }
                    break;

                default:
                    // In all other cases check if the form
                    // has valid values and continue with the
                    // done callback if valid
                    _checkForm(false, false, [$.extend(true, {}, rowdata)]).done(function () {
                        // Empty the delete hidden field in case it was set before
                        $deleteField.val('');

                        var event = null;

                        if (type === 'add') {
                            event = jse.libs.theme.events.WISHLIST_TO_CART();
                        }

                        if (event) {
                            var deferred = $.Deferred();

                            deferred.done(function () {
                                _executeAction($row, $target, rowdata, article, type);
                            });

                            $body.trigger(event, [{ 'deferred': deferred, 'dataset': rowdata }]);
                        } else {
                            _executeAction($row, $target, rowdata, article, type);
                        }
                    }).fail(function () {
                        _cleanupArray(article, $row);
                    });
                    break;
            }
        }
    };

    /**
     * Prevent the submit event that was triggerd
     * by user or by script. If it was triggered
     * by the user, check if it was an "Enter"-key
     * submit from an input field. If so, execute
     * the refresh functionality for that row.
     * If the event was triggered by the script
     * (identified by the data flag "d") check the
     * whole form for errors. Only in case of valid
     * data proceed the submit
     * @param       {object}        e       jQuery event object
     * @param       {boolean}       d       A flag that identifies that the submit was triggered by this script
     * @private
     */
    var _submitHandler = function _submitHandler(e, d) {

        // Prevent the default behaviour
        // in both cases
        e.preventDefault();
        e.stopPropagation();

        if (!d && e.originalEvent) {

            // Check if an input field has triggerd the submit event
            // and call the refresh handler
            var $source = $(e.originalEvent.explicitOriginalTarget);
            if ($source.length && $source.is('input[type="text"]')) {
                $source.closest('.item').find('.button-refresh').first().trigger('click');
            }
        } else if (d) {

            // Check the whole form and only submit
            // it if it's valid
            _checkForm().done(function () {
                // Remove the submit event handler
                // on a successful validation and
                // trigger a submit again, so that the
                // browser executes it's default behavior
                $form.off('submit').trigger('submit');

                // Resolve the deferred if given
                if ((typeof d === 'undefined' ? 'undefined' : _typeof(d)) === 'object') {
                    d.resolve();
                }
            }).fail(function () {
                // Reject the deferred if given
                if ((typeof d === 'undefined' ? 'undefined' : _typeof(d)) === 'object') {
                    d.reject();
                }
            });
        }
    };

    /**
     * Event handler for clicking on the proceed
     * button to get to the checkout process. It
     * checks all items again if they contain valid
     * data. Only if so, proceed
     * @param       {object}        e       jQuery event object
     * @private
     */
    var _submitButtonHandler = function _submitButtonHandler(e) {
        e.preventDefault();
        e.stopPropagation();

        if (isChanged) {
            // Get the complete form data if no row data is given
            var formdata = _generateFormdataObject();

            // Check the formdata for changed values
            $.each(formdata, function () {
                var $changedInput = this.target.find('.' + options.changeClass);

                if ($changedInput) {
                    $changedInput.closest('.item').find('.button-refresh').first().trigger('click');
                }
            });

            isChanged = false;
            _updateChangeState();
            return;
        }

        var $self = $(this),
            destination = $self.attr('href');

        // Check if there is any other process running
        if ($.isEmptyObject(active) && !busy && !updateList) {
            busy = true;

            _checkForm(true, true).done(function () {
                function callback() {
                    location.href = destination;
                }

                jse.libs.hooks.execute(jse.libs.hooks.keys.shop.cart.checkout, { event: e }, 500).then(callback).catch(callback);
            }).always(function () {
                busy = false;
            });
        }
    };

    /**
     * Event handler that checks the form and
     * resolves or rejects the delivered deferred
     * (Used for external payment modules to
     * check if the form is valid)
     * @param       {object}    e               jQuery event object
     * @param       {object}    d               JSON object with the event settings
     * @private
     */
    var _checkFormHandler = function _checkFormHandler(e, d) {
        e.stopPropagation();

        d = d || {};

        _checkForm(d.showChanges, d.revertChanges).done(function () {
            if (d.deferred) {
                d.deferred.resolve();
            }
        }).fail(function () {
            if (d.deferred) {
                d.deferred.reject();
            }
        });
    };

    /**
     * Function that updates the list on focus of
     * the window
     * @private
     */
    var _updateList = function _updateList() {
        updateList = true;
        jse.libs.xhr.ajax({ url: options.updateUrl }, true).done(function (result) {
            // Init with he first line since this ist the heading
            var $lastScanned = $form.find('.order-wishlist .item').first(),
                $target = $();

            // Iterate through the products object and search for the
            // products inside the markup. If the product was found,
            // update the values, if not add the product row at the
            // correct position
            $.each(result.products, function (key, value) {
                var articleId = key.replace('product_', ''),
                    $article = $form.find('input[name="products_id[]"][value="' + articleId + '"]'),
                    $row = null;

                if (!$article.length) {
                    // The article wasn't found on page
                    // -> add it
                    $row = $(value);
                    $row.insertAfter($lastScanned);
                } else {
                    // The article was found on page
                    // -> update it
                    $row = $article.closest('.item');

                    var $qty = $row.find('input[name="cart_quantity[]"]'),
                        oldQty = parseFloat($qty.data().oldValue),
                        currentQty = parseFloat($qty.val()),
                        newQty = parseFloat($(value).find('input[name="cart_quantity[]"]').val());

                    $qty.data('oldValue', newQty);

                    // Add or remove the changed classes depending on
                    // the quantity changes and the on page stored values
                    if (oldQty === currentQty && currentQty !== newQty) {
                        $qty.addClass(options.changeClass);
                    } else if (oldQty !== currentQty && currentQty === newQty) {
                        $qty.removeClass(options.changeClass);
                    }
                }

                $target.add($row);
                $lastScanned = $row;
            });

            // Update the rest of the form
            _updateForm($target, result);
        }).always(function () {
            updateList = false;
        });
    };

    /**
     * Update the input change state
     * @private
     */
    var _updateChangeState = function _updateChangeState() {
        $form.find(options.selectorMapping.submit).text(jse.core.lang.translate(isChanged ? 'refresh' : 'checkout', 'buttons'));
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget
     * @constructor
     */
    module.init = function (done) {

        $updateTarget = $(options.updateTarget);
        $cartEmpty = $(options.cartEmpty);
        $cartNotEmpty = $(options.cartNotEmpty);
        $deleteField = $(options.deleteInput);
        $form = $this.find('form').first();
        deleteFieldName = $deleteField.attr('name');
        action = $form.attr('action');
        transition = { open: true, classOpen: options.classLoading };

        // Sets the current value of the input
        // to an hidden data attribute
        _updateDataValues($form);

        $form.on('input', 'input[type="text"]:not(.gift-coupon-code-input)', _inputHandler).on('blur', 'input[type="text"]:not(.gift-coupon-code-input)', _blurHandler).on('click.delete', '.button-delete', { 'type': 'delete' }, _clickHandler).on('click.refresh', '.button-refresh', { 'type': 'refresh' }, _clickHandler).on('click.addtocart', '.button-to-cart', { 'type': 'add' }, _clickHandler).on('click.submit', '.button-submit', { 'type': 'submit' }, _submitButtonHandler).on('submit', _submitHandler).on(jse.libs.theme.events.CHECK_CART(), _checkFormHandler);

        $('a.toggleusebalance input[name="gv_use_balance"]').on('click', function () {
            console.info('click');location = $(this).parent('a').get(0).href;
        });

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
